;==========================================================================
;Version history:
;==========================================================================
;2010/01/14     First edition
;==========================================================================
;Speech Function Declaration
;==========================================================================
REPEAT_SPEECH:          .EQU    OFF     ;Declare Repeat Speech function
;==========================================================================
        .PAGE0
R_SpeechDPTR:           .DS     2
;==========================================================================
;Speech Status for Each Channel
;==========================================================================
R_ChSpeechStatus:       .DS     D_ChannelNo     ;For each SPU channel
 D_SpeechPlaying:       .EQU    00001000B
 D_SpeechRepeat:        .EQU    00000100B
 D_SpeechPause:         .EQU    00000010B
 D_SpeechZCJump:        .EQU    00000001B
;==========================================================================
;Speech Usage
;==========================================================================
        .DATA
R_SpeechStatus:         .DS     1       ;To display which channel is playing speech
R_SpeechTemp:           .DS     1       ;Temporary for speech usage
R_SpeechIndex:          .DS     2       ;Speech index
R_SpeechIrqCh:          .DS     1       ;Backup channel number for IRQ issued by speech.
R_ChFsLTempS:           .DS     1       ;Speech sampling rate
R_ChFsHTempS:           .DS     1
R_ChDPTRHTempS:         .DS     1       ;Speech type
;==========================================================================
;Repeat Speech
;==========================================================================
        .IF REPEAT_SPEECH = ON
R_RepeatCounter:        .DS     1
        .ENDIF
;==========================================================================
;Purpose: Initialize the RAMs for playing speeches with SPU after power on
;Input: None
;Return: None
;Destroy: A
;==========================================================================
%SpeechPowerUpInitial:  .MACRO
        LDA     #00H
        STA     R_SpeechIndex
        STA     R_SpeechIndex+1
                        .ENDM
;==========================================================================
;Purpose: Initialize the RAMs for playing speeches with SPU after wake-up
;Input: None
;Return: None
;Destroy: A
;==========================================================================
%SpeechWakeUpInitial:   .MACRO
        LDA     #00H
        STA     R_SpeechIndex
        STA     R_SpeechIndex+1
                        .ENDM
;==========================================================================
;Purpose: Pause the speech on designated SPU channel
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X
;==========================================================================
%PauseChVol:    .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitPauseChVol#?
L_#?:
        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable

        LDA     R_ChSpeechStatus,X
        ORA     #D_SpeechPause          ;Set pause flag
        STA     R_ChSpeechStatus,X
L_ExitPauseChVol#?:
                .ENDM
;==========================================================================
;Purpose: Resume the speech on designated SPU channel which is paused by "%PauseChVol"
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X
;==========================================================================
%ResumeChVol:   .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitResumeChVol#?
L_#?:
        LDA     P_SPU_Enable
        ORA     T_ChEnable,X
        STA     P_SPU_Enable

        LDA     R_ChSpeechStatus,X
        AND     #.NOT.(D_SpeechPause)   ;Clear pause flag
        STA     R_ChSpeechStatus,X
L_ExitResumeChVol#?:
                .ENDM
;==========================================================================
;Purpose: Check if the designated channel is paused
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: C = 0, the channel is not paused; C = 1, the channel is paused
;Destroy: A, X
;==========================================================================
%TestChPause:   .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitTestChPause#?
L_#?:
        CLC
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPause          ;Check pause flag
        BEQ     L_ExitTestChPause#?
        SEC                             ;C = 1, pause mode
L_ExitTestChPause#?:
                .ENDM
;==========================================================================
;Purpose: Enable speech repeat mode on designated channel and set the times of repeat mode
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7), Times
;Return: None
;Destroy: A, X, Y
;==========================================================================
%RepeatSpeechOn:        .MACRO  Channel, Times
        LDA     #Times
        STA     R_RepeatCounter

        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitRepeatSpeechOn#?
L_#?:
        LDY     T_ChMap,X
        SEI
        LDA     R_ChSpeechStatus,X
        ORA     #D_SpeechRepeat         ;Set Repeat Speech flag
        STA     R_ChSpeechStatus,X
        CLI

        LDA     P_INT_CtrlH
        ORA     #D_SPUIntEn
        STA     P_INT_CtrlH             ;Enable SPU IRQ

        SEI
        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,X
        STA     P_SPU_IntCtrl           ;Enable channel SPU IRQ
        LDA     R_Ch0VolCtl,Y
        ORA     #D_Bit7
        STA     R_Ch0VolCtl,Y
        STA     R_Ch0VolN,Y
        CLI
L_ExitRepeatSpeechOn#?:
                        .ENDM
;==========================================================================
;Purpose: Disable speech repeat mode on designated channel
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X, Y
;==========================================================================
%RepeatSpeechOff:       .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitRepeatSpeechOff#?
L_#?:
        LDY     T_ChMap,X
        LDA     T_ChEnable,X
        STA     P_SPU_Status            ;Write 1 to clear SPU IRQ status

        LDA     R_Ch0VolCtl,Y
        AND     #~D_Bit7
        STA     R_Ch0VolCtl,Y

        LDA     #00H
        STA     R_RepeatCounter

        LDA     R_ChSpeechStatus,X
        AND     #.NOT.(D_SpeechRepeat)
        STA     R_ChSpeechStatus,X

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X           ;Disable SPU IRQ
        STA     P_SPU_IntCtrl
L_ExitRepeatSpeechOff#?:
                        .ENDM
;==========================================================================
;Purpose: Always repeat the same speech on designated channel
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X, Y
;==========================================================================
%AlwaysRepeatSp:        .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitAlwaysRepeatSp#?
L_#?:
        LDY     T_ChMap,X
        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl           ;Disable SPU IRQ

        LDA     R_Ch0VolCtl,Y
        AND     #~D_Bit7
        STA     R_Ch0VolN,Y

        LDA     R_Ch0VolCtl,Y
        ORA     #D_Bit7
        STA     R_Ch0VolCtl,Y

        LDA     R_Ch0VolN,Y
        ORA     #D_Bit7
        STA     R_Ch0VolN,Y
L_ExitAlwaysRepeatSp#?:
                        .ENDM
;==========================================================================
;Purpose: Stop the speech which is played on designated channel
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X
;==========================================================================
%StopSpeech:            .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitStopSpeech#?
L_#?:
        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable            ;Close channel

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl           ;Disable SPU IRQ

        LDA     P_SPU_Status
        ORA     T_ChEnable,X
        STA     P_SPU_Status            ;Clear IRQ status

        LDA     R_ChSpeechStatus,X
        AND     #.NOT.D_SpeechPlaying
        STA     R_ChSpeechStatus,X

        LDA     R_SpeechStatus
        AND     T_ChDisable,X
        STA     R_SpeechStatus
L_ExitStopSpeech#?:
                        .ENDM
;==========================================================================
;Purpose: Check if a speech is played on designated channel
;Input: Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: C = 0, no speech is played; C = 1, a speech is played
;Destroy: A, X
;==========================================================================
%TestHWSpeech:          .MACRO  Channel
        LDX     #Channel
        CPX     #D_ChannelNo
        BCC     L_#?
        JMP     L_ExitTestHWSpeech#?
L_#?:
        CLC
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BEQ     L_ExitTestHWSpeech#?
        SEC
L_ExitTestHWSpeech#?:
                        .ENDM
;==========================================================================
;Purpose: Set the format of speech which is played by a SPU channel
;Input: Format (ADPCM/PCM)
;Return: None
;Destroy: A
;==========================================================================
%SetSpeechMode:         .MACRO  STR1
        LDA     #00H                    ;Initial and set Bit7 to decide ADPCM or PCM

        .IFSAME    STR1,PCM
        ORA     #D_Bit7                 ;Bit7 = 1, PCM
        .ENDIF

        .IFSAME    STR1,ADPCM
        AND     #~D_Bit7                ;Bit7 = 0, ADPCM
        .ENDIF

        STA     R_ChDPTRHTempS
                        .ENDM
;==========================================================================
%LoadSpeechFreq:        .MACRO  SR
        DW      (65536*SR*16*D_ChannelNo/CPU_Clock)
                        .ENDM
;==========================================================================
